package ga.core.algorithm.util;

import ga.core.individual.IIndividual;
import ga.core.individual.population.IPopulation;

/**
 * Utility class for handling populations.
 * 
 * @since 11.08.2012
 * @author Stephan Dreyer
 */
public final class PopulationUtil {

  /**
   * Instantiation is not allowed.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  private PopulationUtil() {
  }

  /**
   * Gets the maximum fitness of the whole population.
   * 
   * @param pop
   *          The population.
   * @return The maximum fitness.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public static double getMaxFitness(final IPopulation<?> pop) {

    double max = Double.MIN_VALUE;

    for (final IIndividual<?> ind : pop.getIndividuals()) {
      if (ind.isEvaluated() && ind.getFitness() > max) {
        max = ind.getFitness();
      }
    }

    return max;
  }

  /**
   * Gets the minimum fitness of the whole population.
   * 
   * @param pop
   *          The population.
   * @return The minimum fitness.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public static double getMinFitness(final IPopulation<?> pop) {

    double min = Double.MAX_VALUE;

    for (final IIndividual<?> ind : pop.getIndividuals()) {
      if (ind.isEvaluated() && ind.getFitness() < min) {
        min = ind.getFitness();
      }
    }

    return min;
  }

  /**
   * Gets the mean fitness of the whole population.
   * <p />
   * <tt>sum(fit(pop))/length(pop)</tt>
   * 
   * @param pop
   *          The population.
   * @return The mean fitness.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public static double getMeanFitness(final IPopulation<?> pop) {

    double sum = 0d;
    int count = 0;

    for (final IIndividual<?> ind : pop.getIndividuals()) {
      if (ind.isEvaluated()) {
        sum += ind.getFitness();
        count++;
      }
    }

    return sum / count;
  }

  /**
   * Gets the minimum/mean/maximum fitness of the whole population as array. The
   * index of the elements is:
   * <p />
   * 0 -> min <br />
   * 1 -> mean <br />
   * 2 -> max
   * 
   * @param pop
   *          The population.
   * @return The minimum fitness.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public static double[] getMinMeanMaxFitness(final IPopulation<?> pop) {
    double sum = 0d;
    int count = 0;

    double min = Double.MAX_VALUE;
    double max = Double.MIN_VALUE;

    for (final IIndividual<?> ind : pop.getIndividuals()) {
      if (ind.isEvaluated()) {
        sum += ind.getFitness();
        count++;

        if (ind.getFitness() < min) {
          min = ind.getFitness();
        }
        if (ind.getFitness() > max) {
          max = ind.getFitness();
        }
      }
    }

    final double mean = sum / count;

    return new double[] { min, mean, max };
  }
}
